#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_Equipment : BaseTest 
	{

		private BOL.Equipment _tstEQ;
		private DataTable _dtEquipment;
		private BOL.Equipment _blankEQ;
		private DataTable _dtEquipMaint;

		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

			if (this.RefreshData)
			{
				this._dtEquipment = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Equipment WHERE DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", false);
				if (this._dtEquipment.Rows.Count == 0)
				{
					DataTable dtTemp = SprocHelper.GetInsertEquipmentSprocTable(Guid.NewGuid(), DataCreator.GetRandomRecordGuid(TABLES.Address.TableName, ""));
					new Common.StoredProcedure().TransactionalGetValue(Common.VbecsStoredProcs.InsertEquipment.StoredProcName, dtTemp);

					this._dtEquipment= UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Equipment WHERE DivisionCode='" + this.TestDivisionCode + "' AND EquipmentGuid NOT IN (SELECT EquipmentGuid FROM EquipmentMaintenance)", true);
				}

				this._tstEQ = new BOL.Equipment(this._dtEquipment.Rows[0]);

				this._dtEquipMaint = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 EquipmentMaintenance.*, Equipment.DivisionCode FROM EquipmentMaintenance JOIN Equipment ON Equipment.EquipmentGuid = EquipmentMaintenance.EquipmentGuid WHERE DivisionCode = '" + this.TestDivisionCode + "'", false);
				if (this._dtEquipMaint.Rows.Count == 0)
				{
					DataTable dtTemp = SprocHelper.GetInsertEquipmentMaintenanceSprocTable(System.Guid.NewGuid(), this._tstEQ.EquipmentGuid);
					new Common.StoredProcedure().TransactionalGetValue(Common.VbecsStoredProcs.InsertEquipmentMaintenance.StoredProcName, dtTemp);

					this._dtEquipMaint = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 EquipmentMaintenance.*, Equipment.DivisionCode FROM EquipmentMaintenance JOIN Equipment ON Equipment.EquipmentGuid = EquipmentMaintenance.EquipmentGuid WHERE DivisionCode = '" + this.TestDivisionCode + "'", true);
				}
				
				//Testing for Null Columns
				this._blankEQ  = new BOL.Equipment(DAL.Equipment.GetEmptyEquipmentTable().NewRow());

				this.RefreshData = false;
			}


			this._tstEQ = new BOL.Equipment((Guid)this._dtEquipment.Rows[0][TABLES.Equipment.EquipmentGuid]);
			this._tstEQ.SetInitialBrokenRules("FrmNewEquipment",TABLES.Equipment.TableName);
		}

		[TearDown]
		protected void TearDown()
		{

		}

		[Test]
		public void Ctor_GUID_Pass()
		{
			BOL.Equipment testEquip = new BOL.Equipment(this._tstEQ.EquipmentGuid);
			
			Assert.AreEqual(this._tstEQ.Comments, testEquip.Comments, "Comments");
			Assert.AreEqual(this._tstEQ.DivisionCode, testEquip.DivisionCode, "DivisionCode");
			Assert.AreEqual(this._tstEQ.InspectionIndicator, testEquip.InspectionIndicator, "InspectionIndicator");
			Assert.AreEqual(this._tstEQ.InvoiceNumber, testEquip.InvoiceNumber, "InvoiceNumber");
			Assert.AreEqual(this._tstEQ.SerialNumber, testEquip.SerialNumber, "SerialNumber");
			Assert.AreEqual(this._tstEQ.ManufacturerName, testEquip.ManufacturerName, "ManufacturerName");
			Assert.AreEqual(this._tstEQ.ReceiptDate, testEquip.ReceiptDate, "ReceiptDate");
			Assert.AreEqual(this._tstEQ.ReceiptTechId, testEquip.ReceiptTechId, "ReceiptTechId");
			Assert.AreEqual(this._tstEQ.EquipmentGuid, testEquip.EquipmentGuid, "EquipmentGuid");
			Assert.AreEqual(this._tstEQ.EquipmentName, testEquip.EquipmentName, "EquipmentName");
			Assert.AreEqual(this._tstEQ.EquipmentType, testEquip.EquipmentType, "EquipmentType");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void Ctor_GUID_Fail()
		{
			BOL.Equipment testEquip = new BOL.Equipment(System.Guid.Empty);
		}


		[Test]
		public void Ctor_DataRow_Pass()
		{
			BOL.Equipment testEquip = new BOL.Equipment(this._dtEquipment.Rows[0]);

			Assert.AreEqual(this._dtEquipment.Rows[0][TABLES.Equipment.DivisionCode].ToString(), testEquip.DivisionCode, "DivisionCode");
			Assert.AreEqual((System.DateTime) this._dtEquipment.Rows[0][TABLES.Equipment.ReceiptDate], testEquip.ReceiptDate, "ReceiptDate");
			Assert.AreEqual((System.Guid) this._dtEquipment.Rows[0][TABLES.Equipment.EquipmentGuid], testEquip.EquipmentGuid, "EquipmentGuid");
			Assert.AreEqual(this._dtEquipment.Rows[0][TABLES.Equipment.EquipmentName].ToString(), testEquip.EquipmentName, "EquipmentName");
			Assert.AreEqual(this._dtEquipment.Rows[0][TABLES.Equipment.LocalIdentifier].ToString(), testEquip.LocalIdentifier, "LocalIdentifier");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void Ctor_DataRow_Fail()
		{
			DataTable dtTest = new DataTable();
			dtTest.Columns.Add(TABLES.Equipment.EquipmentGuid, typeof(System.Guid));

			BOL.Equipment testEquip = new BOL.Equipment(dtTest.NewRow());
		}


		[Test]
		public void EquipmentName_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstEQ.EquipmentName = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.EquipmentName, "Valid Value");

			//Test for Null Column
			Assert.AreEqual(String.Empty, this._blankEQ.EquipmentName, "Null Column");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void EquipmentName_Fail()
		{
			this._tstEQ.EquipmentName = string.Empty.PadRight(76,'X');
		}

		[Test]
		public void ReceiptDate_Pass()
		{
			System.DateTime tmpValue = BOL.VBECSDateTime.GetDivisionCurrentDateTime();

			this._tstEQ.ReceiptDate = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.ReceiptDate);

			//Test for Null Column
			Assert.AreEqual(System.DateTime.MinValue, this._blankEQ.ReceiptDate, "Null Column");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void ReceiptDate_Fail()
		{
			System.DateTime tmpValue = BOL.VBECSDateTime.GetDivisionCurrentDateTime().AddMinutes(1);

			this._tstEQ.ReceiptDate = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.ReceiptDate);
		}

		[Test]
		public void ReceiptTechID_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstEQ.ReceiptTechId = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.ReceiptTechId);

			//Test for Null Column
			Assert.AreEqual(string.Empty, this._blankEQ.ReceiptTechId, "Null Column");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void ReceiptTechID_Fail()
		{
			this._tstEQ.ReceiptTechId = string.Empty.PadRight(31,'X');
		}

		[Test]
		public void InvoiceNumber_Pass()
		{
			string tmpValue = string.Empty.PadRight(20,'X');

			this._tstEQ.InvoiceNumber = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.InvoiceNumber);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void InvoiceNumber_Fail()
		{
			this._tstEQ.InvoiceNumber = string.Empty.PadRight(21,'X');
		}

		[Test]
		public void LocalIdentifier_Pass()
		{
			string tmpValue = new Random().Next(9999).ToString();

			this._tstEQ.LocalIdentifier = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.LocalIdentifier);

			//Test for Null Column
			Assert.AreEqual(string.Empty, this._blankEQ.LocalIdentifier, "Null Column");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void LocalIdentifier_Fail()
		{
			this._tstEQ.LocalIdentifier = string.Empty.PadRight(26,'X');
		}

		[Test]
		public void MaintenanceContractDate_Pass()
		{
			System.DateTime tmpValue = System.DateTime.Now;

			this._tstEQ.MaintenanceContractDate = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.MaintenanceContractDate);
		}

		[Test]
		public void MaintenanceContractDate_Fail()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			this._tstEQ.IsMaintenanceContractDate = true;
			this._tstEQ.MaintenanceContractDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			brokenRulesBefore = this._tstEQ.BrokenRulesCount;
			//
			this._tstEQ.MaintenanceContractDate = System.DateTime.MinValue;
			brokenRulesAfter = this._tstEQ.BrokenRulesCount;
			//
			Assert.AreEqual(true, brokenRulesAfter > brokenRulesBefore, "BrokenRulesCount");
		}

		[Test]
		public void IsMaintenanceContractDate_Pass()
		{
			bool tmpValue = (!this._tstEQ.IsMaintenanceContractDate);

			this._tstEQ.IsMaintenanceContractDate = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.IsMaintenanceContractDate);
		}

		[Test]
		[Ignore ( StandardIgnoreReasons.NoFailureConditions )]
		public void IsMaintenanceContractDate_Fail()
		{
			//
		}

		[Test]
		public void MaintenanceContractNumber_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstEQ.MaintenanceContractNumber = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.MaintenanceContractNumber);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void MaintenanceContractNumber_Fail()
		{
			this._tstEQ.MaintenanceContractNumber = "".PadRight(26,'X');
		}

		[Test]
		public void InspectionIndicator_Pass()
		{
			bool tmpValue = (!this._tstEQ.InspectionIndicator);

			this._tstEQ.InspectionIndicator = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.InspectionIndicator);

			//Test for Null Column
			Assert.AreEqual(false, this._blankEQ.InspectionIndicator, "Null Column");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void InspectionIndicator_Fail()
		{
		}

		[Test]
		public void OutOfServiceIndicator_Pass()
		{
			bool tmpValue = (!this._tstEQ.OutOfServiceIndicator);

			this._tstEQ.OutOfServiceIndicator = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.OutOfServiceIndicator);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void OutOfServiceIndicator_Fail()
		{
		}

		[Test]
		public void Comments_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstEQ.Comments = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.Comments);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void Comments_Fail()
		{
			this._tstEQ.Comments = string.Empty.PadRight(256,'X');
		}

		[Test]
		public void EquipmentGUID_Pass()
		{
			System.Guid tmpValue = System.Guid.NewGuid();
			
			BOL.Equipment tmpEquipment = new BOL.Equipment(tmpValue);

			Assert.AreEqual(tmpValue, tmpEquipment.EquipmentGuid, "Valid Value");

			//Test for Null Column
			Assert.AreEqual(System.Guid.Empty, this._blankEQ.EquipmentGuid, "Null Column");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.ReadOnlyProperty)]
		public void EquipmentGUID_Fail()
		{
		}

		[Test]
		public void SerialNumber_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstEQ.SerialNumber = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.SerialNumber);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void SerialNumber_Fail()
		{
			this._tstEQ.SerialNumber = "".PadRight(51,'X');
		}

		[Test]
		public void EquipmentType_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstEQ.EquipmentType = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.EquipmentType);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void EquipmentType_Fail()
		{
			this._tstEQ.EquipmentType = "".PadRight(26,'X');
		}

		[Test]
		public void ManufacturerName_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstEQ.ManufacturerName = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.ManufacturerName);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void ManufacturerName_Fail()
		{
			this._tstEQ.ManufacturerName = "".PadRight(51,'X');
		}

		[Test]
		public void ManufacturerAddress_Pass()
		{
			Assert.IsNotNull(this._tstEQ.ManufacturerAddress, "CheckForNull");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.ReadOnlyProperty )]
		public void ManufacturerAddress_Fail()
		{
		}

		[Test]
		public void ManufacturerPhoneNumber_Pass()
		{
			string tmpValue = String.Empty.PadRight(14,'X');

			this._tstEQ.ManufacturerPhoneNumber = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.ManufacturerPhoneNumber);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void ManufacturerPhoneNumber_Fail()
		{
			this._tstEQ.ManufacturerPhoneNumber = "".PadRight(15,'X');
		}

		[Test]
		public void DivisionCode_Pass()
		{
			int tmpValue = new Random().Next(99999);

			this._tstEQ.DivisionCode = tmpValue.ToString();

			Assert.AreEqual(this._tstEQ.DivisionCode, tmpValue.ToString(), "Valid DivisionCode");
			
			//Test for Null Column
            Assert.AreEqual(Common.LogonUser.LogonUserDivisionCode, this._blankEQ.DivisionCode, "Null Column");
		}

		[Test]
		public void DivisionCode_Fail()
		{
			try
			{
				this._tstEQ.DivisionCode = string.Empty;
				Assert.Fail("ArgumentException should have been thrown. DivisionCode = string.Empty");
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true, "DivisionCode = string.Empty");
			}

			
			try
			{
				this._tstEQ.DivisionCode = "".PadRight(6,'X');
				Assert.Fail("ArgumentException should have been thrown. DivisionCode.Length > 5");
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true, "DivisionCode.Length > 5");
			}


		}

		[Test]
		public void NextMaintenanceEvent_Pass()
		{
			string tmpValue = String.Empty.PadRight(255,'X');

			this._tstEQ.NextMaintenanceEvent = tmpValue;

			Assert.AreEqual(tmpValue, this._tstEQ.NextMaintenanceEvent);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void NextMaintenanceEvent_Fail()
		{
			this._tstEQ.NextMaintenanceEvent = "".PadRight(256,'X');
		}

		[Test]
		public void GetEquipmentBySerialNumberAndManufacturer_Pass()
		{
			System.Guid tstValue = BOL.Equipment.GetEquipmentBySerialNumberAndManufacturer(this._tstEQ.SerialNumber, this._tstEQ.ManufacturerName);

			Assert.AreEqual(this._tstEQ.EquipmentGuid, tstValue);
		}

		[Test]
		public void GetEquipmentBySerialNumberAndManufacturer_Fail()
		{
			System.Guid tstValue = BOL.Equipment.GetEquipmentBySerialNumberAndManufacturer("CRACKER", "JAX");

			Assert.AreEqual(System.Guid.Empty, tstValue);
		}

		[Test]
		public void GetEquipment_Pass()
		{
			System.Collections.ArrayList equipments = new System.Collections.ArrayList(1);
			BOL.Equipment equipment = new BOL.Equipment(Guid.NewGuid());
			equipment.EquipmentName = "Unit Test 91";
			equipment.EquipmentType = "Unit Test 91";
			equipment.ManufacturerName = "Unit Test 91";
			equipment.SerialNumber = "Unit Test 91";
			equipment.LocalIdentifier = "Unit Test 91";
			equipment.ReceiptDate = DateTime.Now.AddDays(-1);
			equipment.ReceiptTechId = Environment.UserName;
			equipment.OutOfServiceIndicator = false;
			equipments.Add(equipment);
			BOL.Equipment.InsertEquipment(equipments,Common.UpdateFunction.UnitTests);
			//
			DataTable dt = equipment.GetEquipment();

			if (dt.Rows.Count > 0)
			{
				Assert.AreEqual(dt.Rows[0][TABLES.Equipment.ManufacturerName].ToString(), equipment.ManufacturerName, "ManufacturerName");
			}
			else
			{
				Assert.Fail("No data to test.");
			}
		}

		[Test]
		public void GetEquipment_Fail()
		{
			Common.LogonUser.LogonUserDivisionCode = "XXXXX";
			DataTable dt = this._tstEQ.GetEquipment();

			Assert.AreEqual(0, dt.Rows.Count ,"Rows.count");
		}

		[Test]
		public void GetEquipmentList_Pass()
		{
			System.Collections.ArrayList tmpList = BOL.Equipment.GetEquipmentList();

			Assert.AreEqual(true, tmpList.Count > 0);
		}

		[Test]
		public void GetEquipmentList_Fail()
		{
			Common.LogonUser.LogonUserDivisionCode = "XXXXX";

			System.Collections.ArrayList tmpList = BOL.Equipment.GetEquipmentList();

			Assert.AreEqual(0, tmpList.Count, "Count");
		}

		[Test]
		public void GetEquipmentDetails_Pass()
		{
			System.Collections.ArrayList tmpList = BOL.Equipment.GetEquipmentDetails();

			Assert.AreEqual(true, tmpList.Count > 0);
		}

		[Test]
		public void GetEquipmentDetails_Fail()
		{
			Common.LogonUser.LogonUserDivisionCode = "XXXXX";

			System.Collections.ArrayList tmpList = BOL.Equipment.GetEquipmentDetails();
			System.Collections.ArrayList types = (System.Collections.ArrayList)tmpList[0];
			System.Collections.ArrayList names = (System.Collections.ArrayList)tmpList[1];

			Assert.AreEqual(0, types.Count, "Types.Count");
			Assert.AreEqual(0, names.Count, "Names.Count");
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			System.DateTime tmpValue = System.DateTime.Now;

			this._tstEQ.MaintenanceContractDate = tmpValue;
			this._tstEQ.IsMaintenanceContractDate = true;
			
			DataRow dr = this._tstEQ.LoadDataRowFromThis(this._dtEquipment.Rows[0].Table.NewRow());

			Assert.AreEqual(this._tstEQ.EquipmentName, dr[TABLES.Equipment.EquipmentName].ToString(), "EquipmentName");
			Assert.AreEqual(this._tstEQ.EquipmentType, dr[TABLES.Equipment.EquipmentType].ToString(), "EquipmentType");
			Assert.AreEqual(this._tstEQ.ManufacturerName, dr[TABLES.Equipment.ManufacturerName].ToString(), "ManufacturerName");
			Assert.AreEqual(tmpValue, (System.DateTime)dr[TABLES.Equipment.MaintenanceContractDate], "MaintenanceContractDate-IndicatorOn");

			this._tstEQ.IsMaintenanceContractDate = false;
			dr = this._tstEQ.LoadDataRowFromThis(this._dtEquipment.Rows[0].Table.NewRow());
			Assert.AreEqual(DBNull.Value, dr[TABLES.Equipment.MaintenanceContractDate], "MaintenanceContractDate-IndicatorOff");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void LoadDataRowFromThis_Fail()
		{
		}

		[Test]
		public void GetEmptyEquipmentTable_Pass()
		{
			DataTable dt = BOL.Equipment.GetEmptyEquipmentTable();

			Assert.AreEqual(19, dt.Columns.Count, "Columns.Count");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetEmptyEquipmentTable_Fail()
		{
		}

		[Test]
		public void InsertEquipment_Pass()
		{
			this.RefreshData = true;

			System.Collections.ArrayList al = new System.Collections.ArrayList(1);
			//
			BOL.Equipment equipment = new BOL.Equipment(Guid.NewGuid());
			equipment.EquipmentName = _tstEQ.EquipmentName;
			equipment.EquipmentType = _tstEQ.EquipmentType;
			equipment.SerialNumber = new Random().Next(99999).ToString();
			equipment.ReceiptDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			equipment.ManufacturerName = _tstEQ.ManufacturerName;
			equipment.ManufacturerAddress.AddressLine1 = _tstEQ.ManufacturerAddress.AddressLine1;
			equipment.ManufacturerAddress.City = _tstEQ.ManufacturerAddress.City;
			equipment.ManufacturerAddress.StateCode = _tstEQ.ManufacturerAddress.StateCode;
			equipment.ManufacturerAddress.ZipCode = _tstEQ.ManufacturerAddress.ZipCode;
			//
			al.Add(equipment);

			bool rc = BOL.Equipment.InsertEquipment(al,Common.UpdateFunction.UnitTests);

			Assert.AreEqual(true, rc, "ReturnCode");
		}

		[Test]
		[ExpectedException(typeof(InvalidCastException))]
		public void InsertEquipment_Fail()
		{
			this.RefreshData = true;

			System.Collections.ArrayList al = new System.Collections.ArrayList(1);
			//
			BOL.Equipment equipment = new BOL.Equipment(Guid.NewGuid());
			//ReciptDate is defaulted to 1/1/1 -- this causes the InvalidCastException in the SQL insert
			//	
			al.Add(equipment);

			BOL.Equipment.InsertEquipment(al,Common.UpdateFunction.UnitTests);
		}

		[Test]
		public void Update_Pass()
		{
			this.RefreshData = true;

			_tstEQ.NextMaintenanceEvent = BOL.VBECSDateTime.GetDivisionCurrentDateTime().AddYears(1).ToString();
			
			this._dtEquipMaint.Rows[0][TABLES.EquipmentMaintenance.MaintenanceType] = System.DateTime.Now.ToString();
			this._dtEquipMaint.Rows[0][TABLES.EquipmentMaintenance.MaintenanceResults] = System.DateTime.Now.ToString();
			
			bool rc = _tstEQ.Update(this._dtEquipMaint, Common.UpdateFunction.UnitTests);
			Assert.AreEqual(true, rc, "ReturnCode");
		}

		[Test]
		[ExpectedException(typeof(InvalidCastException))]
		public void Update_Fail()
		{
			this.RefreshData = true;

			BOL.Equipment equipment = new BOL.Equipment(Guid.NewGuid());

			//RecieptDate defaults to 1/1/1 - which causes the InvalidCastException
			equipment.Update(BOL.EquipmentMaintenance.GetEmptyEquipmentMaintenanceTable(),Common.UpdateFunction.UnitTests);
		}

		[Test]
		public void GetEquipmentTypes_Pass()
		{
			System.Collections.ArrayList tmpList = BOL.Equipment.GetEquipmentTypes();

			Assert.AreEqual(true, tmpList.Count > 0, "Count");
		}

		[Test]
		public void GetEquipmentTypes_Fail()
		{
			Common.LogonUser.LogonUserDivisionCode = "XXXXX";

			System.Collections.ArrayList tmpList = BOL.Equipment.GetEquipmentTypes();

			Assert.AreEqual(0, tmpList.Count, "Count");
		}

		[Test]
		public void GetEquipmentManufacturerNames_Pass()
		{
			System.Collections.ArrayList tmpList = BOL.Equipment.GetEquipmentManufacturerNames();

			Assert.AreEqual(true, tmpList.Count > 0, "Count");
		}

		[Test]
		public void GetEquipmentManufacturerNames_Fail()
		{
			Common.LogonUser.LogonUserDivisionCode = "XXXXX";

			System.Collections.ArrayList tmpList = BOL.Equipment.GetEquipmentManufacturerNames();

			Assert.AreEqual(0, tmpList.Count, "Count");
		}

		[Test]
		public void GetManufacturerAddresses_Pass()
		{
			DataTable tmpList = BOL.Equipment.GetManufacturerAddresses(_tstEQ.ManufacturerName);

			Assert.AreEqual(true, tmpList.Rows.Count > 0, "Rows.Count");
		}

		[Test]
		public void GetManufacturerAddresses_Fail()
		{
			DataTable tmpList = BOL.Equipment.GetManufacturerAddresses(string.Empty.PadRight(50,'X'));

			Assert.AreEqual(0, tmpList.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetEquipmentReport_Pass()
		{
			DataSet tmpList = BOL.Equipment.GetEquipmentReport(_tstEQ.EquipmentGuid);

			Assert.AreEqual(true, tmpList.Tables[0].Rows.Count > 0, "Rows.Count");
		}

		[Test]
		public void GetEquipmentReport_Fail()
		{
			DataSet tmpList = BOL.Equipment.GetEquipmentReport(Guid.Empty);

			Assert.AreEqual(0, tmpList.Tables[0].Rows.Count, "Rows.Count");
		}
	}
}
#endif



